<#
    Lenovo - Infrastructure Solutions Group - Windows Drivers Update Utility
    Version 0.60s - 2025-03-04
    Copyright © 2022-present Lenovo. All rights reserved.

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at:

    https://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
#>

$argProvided = $false

if ($args.count -eq 1) {
    $targetUpdateBundleFile = $args[0]
    $argProvided = $true
    $tmpInstallDir = 'LenovoTmpInstallDir'
    $startDirectory = Get-Location
    if (-not(Test-Path -Path .\$targetUpdateBundleFile -PathType Leaf)) {
        Write-Output "Target update file not found"
        Exit 170
    }
    if (-not($targetUpdateBundleFile.split('.')[-1] -eq 'zip')) {
        Write-Output "Target update file not a Lenovo update bundle (.zip)"
        Exit 170
    }
    if (Test-Path -Path $tmpInstallDir) {
        Remove-Item -LiteralPath $tmpInstallDir -Force -Recurse
        New-Item -Path $tmpInstallDir -ItemType directory
    }
    Expand-Archive -Path $targetUpdateBundleFile -DestinationPath $tmpInstallDir -Force
    Set-Location -Path $tmpInstallDir

    if (-not(Test-Path -Path .\index.json -PathType Leaf)) {
        Write-Output "Target update file not a Lenovo update bundle (index.json)"
        Set-Location -Path $startDirectory
        Remove-Item -LiteralPath $tmpInstallDir -Force -Recurse
        Exit 170
    }
}

if (-not(Test-Path -Path .\index.json -PathType Leaf)) {
    Write-Output "Missing metafile file, install script must be started in same directory as 'index.json' file"
    if ($argProvided -eq $true) {
        Set-Location -Path $startDirectory
        Remove-Item -LiteralPath $tmpInstallDir -Force -Recurse
    }
    Exit 170
}

else {
    try {
        $json = Get-Content -Raw -Path .\index.json | ConvertFrom-Json
    }
    catch {
        Write-Output "Error reading index.json file"
        throw $_.Exception.Message
        if ($argProvided -eq $true) {
            Set-Location -Path $startDirectory
            Remove-Item -LiteralPath $tmpInstallDir -Force -Recurse
        }
        Exit 170
    }
}
$rawdata = PnPutil /enum-drivers | Select-Object -Skip 1
$rawdata = $rawdata -join "`r`n" -split "`r`n`r`n"
if ($rawdata -like "*No published driver packages*") {
    $entries = @()
}
else {
    $entries = $rawdata | ForEach-Object {
        $props = $_ -replace ":", "=" | ConvertFrom-StringData
        new-object -TypeName PSCustomObject -Property $props
    }
}
$payloadsToInstall = @()

$windowsDistro = (Get-WmiObject -class Win32_OperatingSystem).Caption
$hciVersion = (Get-WmiObject -class Win32_OperatingSystem).version

if ($windowsDistro -like '* Windows Server*') {
    $windowsDistroVersion = $windowsDistro.split()[3]
    Write-Output "Detected Microsoft Windows Server $windowsDistroVersion"
    $windowsDistroInvVer = "WS" + $windowsDistroVersion
}
elseif ($windowsDistro -like '* Windows *') {
    $windowsDistroVersion = $windowsDistro.split()[2]
    Write-Output "Detected Microsoft Windows Client $windowsDistroVersion"
    $windowsDistroInvVer = "WC" + $windowsDistroVersion
}
elseif ($windowsDistro -like '* Azure Stack HCI') {
    if ($hciVersion -eq "10.0.26100") {
        $windowsDistroVersion = '2025'
        $windowsDistroInvVer = "WS" + $windowsDistroVersion
        Write-Output "Detected Microsoft Azure Stack HCI 24H2"
    } 
    elseif ($hciVersion -eq "10.0.25398") {
        $windowsDistroVersion = '2022'
        $windowsDistroInvVer = "WS" + $windowsDistroVersion
        Write-Output "Detected Microsoft Azure Stack HCI 23H2"
    }
    elseif ($hciVersion -eq "10.0.20349") {
        $windowsDistroVersion = '2022'
        $windowsDistroInvVer = "WS" + $windowsDistroVersion
        Write-Output "Detected Microsoft Azure Stack HCI 22H2"
    }
    else {
        $windowsDistroVersion = '2022'
        $windowsDistroInvVer = "WS" + $windowsDistroVersion
        Write-Output "Detected Microsoft Azure Stack HCI 22H2"
    }
}
else {
    Write-Output "Non-supported OS - ($windowsDistro)"
    if ($argProvided -eq $true) {
        Set-Location -Path $startDirectory
        Remove-Item -LiteralPath $tmpInstallDir -Force -Recurse
    }
    Exit 171
}

$serverManufacturer = Get-WmiObject Win32_BaseBoard | Select-Object -ExpandProperty Manufacturer
if ($serverManufacturer.ToLower() -ne "lenovo") {
    Write-Output "Non-supported Server Manufacturer: $serverManufacturer"
    if ($argProvided -eq $true) {
        Set-Location -Path $startDirectory
        Remove-Item -LiteralPath $tmpInstallDir -Force -Recurse
    }
    Exit 171
}

$mtModel = Get-WmiObject Win32_ComputerSystem | Select-Object -ExpandProperty SystemSKUNumber
$mt = $mtModel.Substring(0, 4)

$mtSupported = $false

foreach ($applicableMachineType in $json.Oem.ApplicableMachineTypes) {
    if ($applicableMachineType -eq $mt) {
        $mtSupported = $true
    }
}
if ($mtSupported -eq $false) {
    Write-Output "Non-supported Server Machine Type: $mt"
    if ($argProvided -eq $true) {
        Set-Location -Path $startDirectory
        Remove-Item -LiteralPath $tmpInstallDir -Force -Recurse
    }
    Exit 171
}

Write-Output "Supported Lenovo Machine Type Found:  $mt"

$devices = Get-PnpDevice | where-object { $_.InstanceID.StartsWith('PCI') }
$updateRequired = $false
foreach ($device in $devices) {
    $instanceID = $device.InstanceID
    $pciVendor = $instanceID.Substring(8, 4)
    $pciDevice = $instanceID.Substring(17, 4)
    $pciSubVendor = $instanceID.Substring(33, 4)
    $pciSubDevice = $instanceID.Substring(29, 4)
    $pciId = $pciVendor + "-" + $pciDevice + "-" + $pciSubVendor + "-" + $pciSubDevice
    foreach ($miniFixId in $json.Updates) {
        foreach ($invElement in $miniFixId.Oem.Inventory) {
            if ($invElement.OperatingSystem -eq $windowsDistroInvVer) {
                if ($payloadsToInstall -notcontains $miniFixId.Payload) {
                    $invPciId = $invElement.SoftwareId.split('_')[-1].substring(0, 19)
                    if ( ($invPciId -eq $pciId) -or (($invPciId.split('-')[2] -eq "0000") -and (($invPciId.split('-')[0] -eq $pciVendor) -and ($invPciId.split('-')[1] -eq $pciDevice))) ) {
                        Write-Output "Discovered potential appectable device:"
                        Write-Output "`tDiscovered device PCI ID:  $pciId"
                        Write-Output "`tPotential Update SoftwareId: $($invElement.SoftwareId)"
                        $splitSoftwareId = $invElement.SoftwareId.Split('_')
                        $splitSoftwareId = ($splitSoftwareId | Select-Object -skiplast 1) -join '_'
                        $splitSoftwareId = $splitSoftwareId.Split('-')
                        $oobDriverInf = ($splitSoftwareId | Select-Object -skip 1) -join '-'
                        Write-Output "`tOut of box driver: $oobDriverInf (update provides: $($invElement.Version))"
                        $installDriver = $true
                        $oobInstalled = $false
                        foreach ($entry in $entries) {
                            $installedSoftwareId = $entry."Original Name".Substring(0, $entry."Original Name".Length - 4).toLower()
                            $installedVersion = $entry."Driver Version".Split(" ", 2)[-1]
                            if ( $installedSoftwareId -eq $oobDriverInf ) {
                                Write-Output "`tOut of box driver installed, version: $($installedVersion)"
                                $oobInstalled = $true
                                if ( $installedVersion -gt $invElement.Version ) {
                                    $installDriver = $false
                                    Write-Output "`tNo update required"
                                }
                            }
                        }
                        if ( $installDriver ) {
                            if ( -not $oobInstalled ) {
                                Write-Output "`tOut of box driver not installed"
                            }
                            $payloadsToInstall += $miniFixId.Payload
                            Write-Output "`tUpdate Required ( $($miniFixId.Payload) )"
                            $updateRequired = $true
                        }
                    }
                }   
            }
        }
    }
}
## ACPI drivers enablement
foreach ($miniFixId in $json.Updates) {
    foreach ($invElement in $miniFixId.Oem.Inventory) {
        if ($invElement.OperatingSystem -eq $windowsDistroInvVer) {
            $invPciId = $invElement.SoftwareId.split('_')[-1].substring(0, 19)
            if ($invPciId -eq "0000-0000-0000-0000") {
                if ($payloadsToInstall -notcontains $miniFixId.Payload) {
                    Write-Output "`tDiscovered Potential Update SoftwareId: $($invElement.SoftwareId)"
                    $splitSoftwareId = $invElement.SoftwareId.Split('_')
                    $splitSoftwareId = ($splitSoftwareId | Select-Object -skiplast 1) -join '_'
                    $splitSoftwareId = $splitSoftwareId.Split('-')
                    $oobDriverInf = ($splitSoftwareId | Select-Object -skip 1) -join '-'
                    Write-Output "`tOut of box driver: $oobDriverInf (update provides: $($invElement.Version))"
                    $installDriver = $true
                    $oobInstalled = $false
                    foreach ($entry in $entries) {
                        $installedSoftwareId = $entry."Original Name".Substring(0, $entry."Original Name".Length - 4).toLower()
                        $installedVersion = $entry."Driver Version".Split(" ", 2)[-1]
                        if ( $installedSoftwareId -eq $oobDriverInf ) {
                            Write-Output "`tOut of box driver installed, version: $($installedVersion)"
                            $oobInstalled = $true
                            if ( $installedVersion -gt $invElement.Version ) {
                                $installDriver = $false
                                Write-Output "`tNo update required"
                            }
                        }
                    }
                    if ( $installDriver ) {
                        if ( -not $oobInstalled ) {
                            Write-Output "`tOut of box driver not installed"
                        }
                        $payloadsToInstall += $miniFixId.Payload
                        Write-Output "`tUpdate Required ( $($miniFixId.Payload) )"
                        $updateRequired = $true
                    }
                }
            }
        }
    }
}

## Special case for AMD Chipset
if ($json.Oem.Name -match "chipset.sp6t") {
    $CPUid=(Get-CimInstance -ClassName Win32_Processor | Select-Object -ExpandProperty Level)
    if ( $CPUid -ne 26 ) {
        $payloadsToInstall=$payloadsToInstall | Where-Object { $_ -notmatch "amdhsmp" }
    }
}

if ( $updateRequired) {
    Write-Output "`r`nPayloads to install:`r`n $payloadsToInstall"
    if (Test-Path install) {
        Remove-Item 'install' -Force -Recurse -ErrorAction SilentlyContinue
    }
    New-Item install -itemtype directory | Out-Null
    foreach ($payloadToInstall in $payloadsToInstall) {
        $installPath = "install\" + $payloadToInstall.split("/")[2].Replace(".zip", "")
        Expand-Archive $payloadToInstall -DestinationPath $installPath -Force
    }
    Write-Output "`r`nStarting Install"
    Set-Location install
    Start-Process -WindowStyle hidden pnputil -argumentlist "/add-driver .\*.inf /subdirs /install"
    Set-Location ..\
    Write-Output "`r`nUpdates completed"
}
else {
    Write-Output "`r`nNo update Required"
}

if ($argProvided -eq $true) {
    Set-Location -Path $startDirectory
    Start-Sleep -Seconds 5
    Remove-Item -LiteralPath $tmpInstallDir -Force -Recurse
}
Exit 0
# SIG # Begin signature block
# MIIoLQYJKoZIhvcNAQcCoIIoHjCCKBoCAQExDzANBglghkgBZQMEAgIFADCBiQYK
# KwYBBAGCNwIBBKB7MHkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTS
# xil+81ECAQACAQACAQACAQACAQAwQTANBglghkgBZQMEAgIFAAQwWQhjq6yxKp4Q
# BqvwYwmOddG7Az5YWlW749j1B/4KPIpBe1M1SrBfdEXGX0gXEJ+ToIINjjCCBrAw
# ggSYoAMCAQICEAitQLJg0pxMn17Nqb2TrtkwDQYJKoZIhvcNAQEMBQAwYjELMAkG
# A1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRp
# Z2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4X
# DTIxMDQyOTAwMDAwMFoXDTM2MDQyODIzNTk1OVowaTELMAkGA1UEBhMCVVMxFzAV
# BgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVk
# IEc0IENvZGUgU2lnbmluZyBSU0E0MDk2IFNIQTM4NCAyMDIxIENBMTCCAiIwDQYJ
# KoZIhvcNAQEBBQADggIPADCCAgoCggIBANW0L0LQKK14t13VOVkbsYhC9TOM6z2B
# l3DFu8SFJjCfpI5o2Fz16zQkB+FLT9N4Q/QX1x7a+dLVZxpSTw6hV/yImcGRzIED
# Pk1wJGSzjeIIfTR9TIBXEmtDmpnyxTsf8u/LR1oTpkyzASAl8xDTi7L7CPCK4J0J
# wGWn+piASTWHPVEZ6JAheEUuoZ8s4RjCGszF7pNJcEIyj/vG6hzzZWiRok1MghFI
# UmjeEL0UV13oGBNlxX+yT4UsSKRWhDXW+S6cqgAV0Tf+GgaUwnzI6hsy5srC9Kej
# Aw50pa85tqtgEuPo1rn3MeHcreQYoNjBI0dHs6EPbqOrbZgGgxu3amct0r1EGpIQ
# gY+wOwnXx5syWsL/amBUi0nBk+3htFzgb+sm+YzVsvk4EObqzpH1vtP7b5NhNFy8
# k0UogzYqZihfsHPOiyYlBrKD1Fz2FRlM7WLgXjPy6OjsCqewAyuRsjZ5vvetCB51
# pmXMu+NIUPN3kRr+21CiRshhWJj1fAIWPIMorTmG7NS3DVPQ+EfmdTCN7DCTdhSm
# W0tddGFNPxKRdt6/WMtyEClB8NXFbSZ2aBFBE1ia3CYrAfSJTVnbeM+BSj5AR1/J
# gVBzhRAjIVlgimRUwcwhGug4GXxmHM14OEUwmU//Y09Mu6oNCFNBfFg9R7P6tuyM
# MgkCzGw8DFYRAgMBAAGjggFZMIIBVTASBgNVHRMBAf8ECDAGAQH/AgEAMB0GA1Ud
# DgQWBBRoN+Drtjv4XxGG+/5hewiIZfROQjAfBgNVHSMEGDAWgBTs1+OC0nFdZEzf
# Lmc/57qYrhwPTzAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# dwYIKwYBBQUHAQEEazBpMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2Vy
# dC5jb20wQQYIKwYBBQUHMAKGNWh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9E
# aWdpQ2VydFRydXN0ZWRSb290RzQuY3J0MEMGA1UdHwQ8MDowOKA2oDSGMmh0dHA6
# Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3JsMBwG
# A1UdIAQVMBMwBwYFZ4EMAQMwCAYGZ4EMAQQBMA0GCSqGSIb3DQEBDAUAA4ICAQA6
# I0Q9jQh27o+8OpnTVuACGqX4SDTzLLbmdGb3lHKxAMqvbDAnExKekESfS/2eo3wm
# 1Te8Ol1IbZXVP0n0J7sWgUVQ/Zy9toXgdn43ccsi91qqkM/1k2rj6yDR1VB5iJqK
# isG2vaFIGH7c2IAaERkYzWGZgVb2yeN258TkG19D+D6U/3Y5PZ7Umc9K3SjrXyah
# lVhI1Rr+1yc//ZDRdobdHLBgXPMNqO7giaG9OeE4Ttpuuzad++UhU1rDyulq8aI+
# 20O4M8hPOBSSmfXdzlRt2V0CFB9AM3wD4pWywiF1c1LLRtjENByipUuNzW92NyyF
# PxrOJukYvpAHsEN/lYgggnDwzMrv/Sk1XB+JOFX3N4qLCaHLC+kxGv8uGVw5ceG+
# nKcKBtYmZ7eS5k5f3nqsSc8upHSSrds8pJyGH+PBVhsrI/+PteqIe3Br5qC6/To/
# RabE6BaRUotBwEiES5ZNq0RA443wFSjO7fEYVgcqLxDEDAhkPDOPriiMPMuPiAsN
# vzv0zh57ju+168u38HcT5ucoP6wSrqUvImxB+YJcFWbMbA7KxYbD9iYzDAdLoNMH
# AmpqQDBISzSoUSC7rRuFCOJZDW3KBVAr6kocnqX9oKcfBnTn8tZSkP2vhUgh+Vc7
# tJwD7YZF9LRhbr9o4iZghurIr6n+lB3nYxs6hlZ4TjCCBtYwggS+oAMCAQICEAKo
# ZtwHoYwxdseYPWEHhC0wDQYJKoZIhvcNAQELBQAwaTELMAkGA1UEBhMCVVMxFzAV
# BgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2VydCBUcnVzdGVk
# IEc0IENvZGUgU2lnbmluZyBSU0E0MDk2IFNIQTM4NCAyMDIxIENBMTAeFw0yNDA4
# MjcwMDAwMDBaFw0yNTA4MjcyMzU5NTlaMF4xCzAJBgNVBAYTAlVTMRcwFQYDVQQI
# Ew5Ob3J0aCBDYXJvbGluYTEUMBIGA1UEBxMLTW9ycmlzdmlsbGUxDzANBgNVBAoT
# Bkxlbm92bzEPMA0GA1UEAxMGTGVub3ZvMIIBojANBgkqhkiG9w0BAQEFAAOCAY8A
# MIIBigKCAYEAoXMkBo1qx4mqpGyaZb0kXAwhHqxv5R4Y6csuf8pyT0dg+rMm29+X
# cgf0MXDxmsqa9qRW2OioMYexuJAUZ5V8I2F7wy8HI1kiKDw2cXaWC7W1YmIJseLa
# fjRPE8CGNi7jzoJxvvxcMjnw6PwZdOGxXWPSa64ADnq8n/DbWCRBR64LRLySCOda
# VlsdwpAHDnnWCS7+qouSZsbwhoM7poF0GdLOfS8ZnjdwnT89+JdsuTJlRm0VuEav
# x4uoaz1KkgyJ8KnuYaS+U1xN8J3b0FA2TKujKtvA3xDcpGJB6Hmk/R/jbDOi2vUR
# 5P/dIIPbiYKGyz6VOd6kGAAj1HfUPS5kcDqy9cKT2yQDGGmD+KhDdx8ad1P3ZWCD
# brwibOwuQhsC2wyV6umMD7r6VHSpByKdk6y4hWnl+RVNbzXA6nKzBY4XU1vVLBxe
# qZY6zHj2VCbK/N/7NMb3C8U15YgMCybqj+YI4mnykABOivpRyqb1QlE26vohmXt2
# 8rTKmTnpOahHAgMBAAGjggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5h
# ewiIZfROQjAdBgNVHQ4EFgQUTLjGFagvuRxcsnDLJTay9K40mMMwPgYDVR0gBDcw
# NTAzBgZngQwBBAEwKTAnBggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5j
# b20vQ1BTMA4GA1UdDwEB/wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYD
# VR0fBIGtMIGqMFOgUaBPhk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNl
# cnRUcnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBT
# oFGgT4ZNaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0
# Q29kZVNpZ25pbmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEB
# BIGHMIGEMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYI
# KwYBBQUHMAKGUGh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRy
# dXN0ZWRHNENvZGVTaWduaW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1Ud
# EwQCMAAwDQYJKoZIhvcNAQELBQADggIBALAKrhuJNNlt887rmj87/PUEA4mYtZA7
# 8AyY6gnMYHGayhs/uZ/ZJ3YZIXTAWWjmhoSXHKjtnwL6wQFcMuoZkTKDrOuTGysC
# K1m5qR8Nicw+mYrGipjNGpviriiDhU/W4E1lL06OS6HY6+GM7wiRBPQmZGM6pbGp
# 7GtGPIiUYT3hQnADuEHzgWOzQOpsNwQe27y34/ElG8yR7b1An610m5MEnTGPax1u
# MddhQzVXhKsuIj/c9WxOnABFPE0GCEgVKOZiRUExR4Jvrv6Zu2xj/YaGUtqEFrsK
# aysey1rKbe95+vX3AymkUwC0Wv3aTvTMsbE9nmGX+yhiMxXoa+pOWkcQ3IMncMJh
# pwwr+3oz3otCg0Qo/89/TtHcp8SYF448G8rgIl7vlvOiD7zeL939E/3pMchhIeHG
# dL1Yr2ud0bl2nP3/KzBCDWXWmc8b4KQb3nh6CPM2iTox5+ChdN2mcrvN/ns064WK
# oqPjXMcN+DEEj08wMFACZM521tQjT6LkFFXYcZ4n4UTv7SeGUOUwA0bsqQKieK8b
# OsMXPfvzIogm8x3cAem0YuEsUOsXBfCc668G3rG5n7oRs0Htyao2+D2Xg4tH74LK
# ITwSQe19ojVxIPBgHIQ1MnHi4o45MhdweebILhqMUZEesGL0rCKYX7ijRaLvKRaG
# 7TyThOnKNYRHMYIZ5DCCGeACAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMO
# RGlnaUNlcnQsIEluYy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29k
# ZSBTaWduaW5nIFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExAhACqGbcB6GMMXbHmD1h
# B4QtMA0GCWCGSAFlAwQCAgUAoIGMMBAGCisGAQQBgjcCAQwxAjAAMBkGCSqGSIb3
# DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
# MD8GCSqGSIb3DQEJBDEyBDBrGIaKGxoOiPQABzowLY361vXEGrjL48ge/olgNImj
# k/cjoUHYWFZrOzk6zbvG2XkwDQYJKoZIhvcNAQEBBQAEggGAmEgMbgowET7DFpRI
# bAEVGtRNEkHMQjF9E8gYYiXc6n7St1ADHXJV7P2bCw1Cl7M9tSJC1DsbdmecOksL
# kLKzjHWQt9l0yAdfqimVUzHXlH1zrpf8nEVVr1S6m3cUh2n5B7HAKidHXr06QQm2
# G66bBPjddNI1VR4bPegO+RDGWWVABoGwH3ZH6Z9WecAQGTdOS/eR+uIdBFWEgtCF
# JCop6fL2p//ISa4D1X/97hF7qXmLPJIbOujZVsPJ2ZAPbioMpQFmkzXDc1JfkXyI
# pGfgr4jwc75AG3rnraQ+AddLpIciRyiLmjpNxdlSzIIGAwCxMmNN0CKDvqmoTbwh
# +UzKCfRR1+W1BxYIF+IoBwj8coLZ1GwohEzonZ6jKF7fdFao7J5UUhSNb7ERNZza
# pO4ba+88A+WLv5Lf0/wk8C7FeisqszLE7mV8GsCAbTfbU12PdNzO+0ChZs3Nd5NN
# RBg6wmdSnUQyrUbTeFYlJBeHFBRMhqyoQF9jrVGto+qwdU+coYIXKTCCFyUGCisG
# AQQBgjcDAwExghcVMIIXEQYJKoZIhvcNAQcCoIIXAjCCFv4CAQMxDzANBglghkgB
# ZQMEAgEFADBnBgsqhkiG9w0BCRABBKBYBFYwVAIBAQYJYIZIAYb9bAcBMCEwCQYF
# Kw4DAhoFAAQU6QwgML6pSLIQyyzr6W4iGicL0BUCEAKRP6AEM5pEl6DY6fV9EfwY
# DzIwMjUwMzA0MTYyNTQ3WqCCEwMwgga8MIIEpKADAgECAhALrma8Wrp/lYfG+ekE
# 4zMEMA0GCSqGSIb3DQEBCwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdp
# Q2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2
# IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwHhcNMjQwOTI2MDAwMDAwWhcNMzUxMTI1
# MjM1OTU5WjBCMQswCQYDVQQGEwJVUzERMA8GA1UEChMIRGlnaUNlcnQxIDAeBgNV
# BAMTF0RpZ2lDZXJ0IFRpbWVzdGFtcCAyMDI0MIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEAvmpzn/aVIauWMLpbbeZZo7Xo/ZEfGMSIO2qZ46XB/QowIEMS
# vgjEdEZ3v4vrrTHleW1JWGErrjOL0J4L0HqVR1czSzvUQ5xF7z4IQmn7dHY7yijv
# oQ7ujm0u6yXF2v1CrzZopykD07/9fpAT4BxpT9vJoJqAsP8YuhRvflJ9YeHjes4f
# duksTHulntq9WelRWY++TFPxzZrbILRYynyEy7rS1lHQKFpXvo2GePfsMRhNf1F4
# 1nyEg5h7iOXv+vjX0K8RhUisfqw3TTLHj1uhS66YX2LZPxS4oaf33rp9HlfqSBeP
# ejlYeEdU740GKQM7SaVSH3TbBL8R6HwX9QVpGnXPlKdE4fBIn5BBFnV+KwPxRNUN
# K6lYk2y1WSKour4hJN0SMkoaNV8hyyADiX1xuTxKaXN12HgR+8WulU2d6zhzXomJ
# 2PleI9V2yfmfXSPGYanGgxzqI+ShoOGLomMd3mJt92nm7Mheng/TBeSA2z4I78Jp
# wGpTRHiT7yHqBiV2ngUIyCtd0pZ8zg3S7bk4QC4RrcnKJ3FbjyPAGogmoiZ33c1H
# G93Vp6lJ415ERcC7bFQMRbxqrMVANiav1k425zYyFMyLNyE1QulQSgDpW9rtvVcI
# H7WvG9sqYup9j8z9J1XqbBZPJ5XLln8mS8wWmdDLnBHXgYly/p1DhoQo5fkCAwEA
# AaOCAYswggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB
# /wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwH
# ATAfBgNVHSMEGDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4EFgQUn1cs
# A3cOKBWQZqVjXu5Pkh92oFswWgYDVR0fBFMwUTBPoE2gS4ZJaHR0cDovL2NybDMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVT
# dGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEEgYMwgYAwJAYIKwYBBQUHMAGGGGh0
# dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2NhY2Vy
# dHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRp
# bWVTdGFtcGluZ0NBLmNydDANBgkqhkiG9w0BAQsFAAOCAgEAPa0eH3aZW+M4hBJH
# 2UOR9hHbm04IHdEoT8/T3HuBSyZeq3jSi5GXeWP7xCKhVireKCnCs+8GZl2uVYFv
# Qe+pPTScVJeCZSsMo1JCoZN2mMew/L4tpqVNbSpWO9QGFwfMEy60HofN6V51sMLM
# XNTLfhVqs+e8haupWiArSozyAmGH/6oMQAh078qRh6wvJNU6gnh5OruCP1QUAvVS
# u4kqVOcJVozZR5RRb/zPd++PGE3qF1P3xWvYViUJLsxtvge/mzA75oBfFZSbdakH
# Je2BVDGIGVNVjOp8sNt70+kEoMF+T6tptMUNlehSR7vM+C13v9+9ZOUKzfRUAYSy
# yEmYtsnpltD/GWX8eM70ls1V6QG/ZOB6b6Yum1HvIiulqJ1Elesj5TMHq8CWT/xr
# W7twipXTJ5/i5pkU5E16RSBAdOp12aw8IQhhA/vEbFkEiF2abhuFixUDobZaA0Vh
# qAsMHOmaT3XThZDNi5U2zHKhUs5uHHdG6BoQau75KiNbh0c+hatSF+02kULkftAR
# jsyEpHKsF7u5zKRbt5oK5YGwFvgc4pEVUNytmB3BpIiowOIIuDgP5M9WArHYSAR1
# 6gc0dP2XdkMEP5eBsX7bf/MGN4K3HP50v/01ZHo/Z5lGLvNwQ7XHBx1yomzLP8lx
# 4Q1zZKDyHcp4VQJLu2kWTsKsOqQwggauMIIElqADAgECAhAHNje3JFR82Ees/Shm
# Kl5bMA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
# Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERp
# Z2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIy
# MzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7
# MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1l
# U3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDGhjUG
# SbPBPXJJUVXHJQPE8pE3qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI82j6ffOc
# iQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR8XOxs+4rgISKIhjf69o9xBd/qxkr
# PkLcZ47qUT3w1lbU5ygt69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ3HxqV3rw
# N3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43IbtArF+y3kp9zvU5EmfvDqVjbOSm
# xR3NNg1c1eYbqMFkdECnwHLFuk4fsbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu
# 9Yemj052FVUmcJgmf6AaRyBD40NjgHt1biclkJg6OBGz9vae5jtb7IHeIhTZgirH
# kr+g3uM+onP65x9abJTyUpURK1h0QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506
# o9UD4L/wojzKQtwYSH8UNM/STKvvmz3+DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklN
# iyDSLFc1eSuo80VgvCONWPfcYd6T/jnA+bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGT
# yYwMO1uKIqjBJgj5FBASA31fI7tk42PgpuE+9sJ0sj8eCXbsq11GdeJgo1gJASgA
# DoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TNsQIDAQABo4IBXTCCAVkwEgYDVR0T
# AQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+eyG8wHwYD
# VR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMG
# A1UdJQQMMAoGCCsGAQUFBwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYY
# aHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAChjVodHRwOi8vY2Fj
# ZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNV
# HR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRU
# cnVzdGVkUm9vdEc0LmNybDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1s
# BwEwDQYJKoZIhvcNAQELBQADggIBAH1ZjsCTtm+YqUQiAX5m1tghQuGwGC4QTRPP
# MFPOvxj7x1Bd4ksp+3CKDaopafxpwc8dB+k+YMjYC+VcW9dth/qEICU0MWfNthKW
# b8RQTGIdDAiCqBa9qVbPFXONASIlzpVpP0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpP
# kWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8dU+6WvepELJd6f8oVInw1YpxdmXa
# zPByoyP6wCeCRK6ZJxurJB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY+/umnXKv
# xMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl6
# 3f8lY5knLD0/a6fxZsNBzU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhzq6YB
# T70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j/R0o08f56PGYX/sr2H7yRp11LB4n
# LCbbbxV7HhmLNriT1ObyF5lZynDwN7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvt
# lUG3OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm
# 2fBldkKmKYcJRyvmfxqkhQ/8mJb2VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqh
# K/bt1nz8MIIFjTCCBHWgAwIBAgIQDpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0B
# AQwFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMjIwODAxMDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz
# 7MKnJS7JIT3yithZwuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS
# 5F/WBTxSD1Ifxp4VpX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7
# bXHiLQwb7iDVySAdYyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfI
# SKhmV1efVFiODCu3T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jH
# trHEtWoYOAMQjdjUN6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14
# Ztk6MUSaM0C/CNdaSaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2
# h4mXaXpI8OCiEhtmmnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt
# 6zPZxd9LBADMfRyVw4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPR
# iQfhvbfmQ6QYuKZ3AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ER
# ElvlEFDrMcXKchYiCd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4K
# Jpn15GkvmB0t9dmpsh3lGwIDAQABo4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAd
# BgNVHQ4EFgQU7NfjgtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SS
# y4IxLVGLp6chnfNtyA8wDgYDVR0PAQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAC
# hjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURS
# b290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRV
# HSAAMA0GCSqGSIb3DQEBDAUAA4IBAQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyh
# hyzshV6pGrsi+IcaaVQi7aSId229GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO
# 0Cre+i1Wz/n096wwepqLsl7Uz9FDRJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo
# 8L8vC6bp8jQ87PcDx4eo0kxAGTVGamlUsLihVo7spNU96LHc/RzY9HdaXFSMb++h
# UD38dglohJ9vytsgjTVgHAIDyyCwrFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5x
# aiNrIv8SuFQtJ37YOtnwtoeW/VvRXKwYw02fc7cBqZ9Xql4o4rmUMYIDdjCCA3IC
# AQEwdzBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5
# BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0
# YW1waW5nIENBAhALrma8Wrp/lYfG+ekE4zMEMA0GCWCGSAFlAwQCAQUAoIHRMBoG
# CSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjUwMzA0
# MTYyNTQ3WjArBgsqhkiG9w0BCRACDDEcMBowGDAWBBTb04XuYtvSPnvk9nFIUIck
# 1YZbRTAvBgkqhkiG9w0BCQQxIgQgkxcxWMrOa5PHJ1AFcN0YNQGyBiuQxaKApYOY
# 5OYnXcEwNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgdnafqPJjLx9DCzojMK7WVnX+
# 13PbBdZluQWTmEOPmtswDQYJKoZIhvcNAQEBBQAEggIAYMtW6Pw8vTuz2kr6uXOH
# tBjTPIaMKi1LgwDiMc6VBsSb4goN3YqAPSJgmuygzzfLG4X0uv8smPlmCFX61jT8
# 1HuA6IPouBx1QDqzn9RlmdmcSSbe/mG2Mu+Z6E5cxLg2T932v8LqT0/KwLxLfgYg
# DfAg59fiqvfdAF1hbqihc/WGHSyfM77gZMvBICtEk9qNpdhXux+nX5RYj+EsOmNw
# iWzS5xopmTRZ/avKFnNku8GK5837L22gu1cGJ1GZTkTWcQ5u4FTNXUYnIcy8L6Rn
# dWy8PKTOqD60M6iaZlIrnPS9fO2vEhqK7RugfX7MfSTWXCrCzFZBScK00eRo81/Z
# BNXOngVYVBUbB0XjFHI3N6Be+nm7Ot08a6MtpQztbem/PMX96Jx/Hdl4mBDPDNFV
# EDZSsKoEgzx9tZt9NvyyAXxu4d4sUa4trqynqfhN3mUHzxsXLUdJdmcihxlNb0gY
# SoiFQfen0lfwtQFxtDzfpHmN9CgoEYSF9gDoRDn0Oowh60mtv47i91nuUt81ThCP
# kRvt2lMcKJ16oWqP37+SzmRw3XNGq2ISicHGy9GW+LTleMffDSCk4cvbm0Hoe1js
# SRgW8rR9GVbELB1hdOe3YT5rzyCyTW7eKPnqg2m1EOGUijI0/2zUvSIkf6S9f7i6
# BVylmNx5ifI9h3yCEhmInyk=
# SIG # End signature block
